%{
This m-file belongs to Chapter 4 of the dissertation 'Electrochemical CO2
conversion with a flow-through copper hollow fibre'
Anne Sustronk (1,2)
(1) Films in Fluids, Department of Science and Technology, University of 
Twente, Enschede, The Netherlands
(2) Photocatalytic Synthesis Group, Department of Science and Technology, 
University of Twente, Enschede, The Netherlands

The file is used to prepare Figures 2 and 3 in the chapter.

This file is called by Fig2And3.m and requires CalcErr.m for execution.

Note that the file locations in lines 46, 73, 78 and 83 should be added 
before execution of Fig2And3.m.

Brief overview of experimental conditions: Cu hollow fiber working 
electrode, 0.5M KHCO3 or KOH  in MilliQ water, CO purged through fiber,
Ar purged through head space, chronoamperometry

Note that the flow rate to calculate the partial current densities is set
to 30 ml min-1. This is most likely an underestimation of the actual flow
rate at more negative potentials

Note that the formation of formate most likely occurs through a chemical 
rather than an electrochemical reaction. For this reason the concentration
FA is calculated

Version date m-file: March 22, 2023
%}

function [Data] = ConstantPotentialCO(Data)
warning('off','MATLAB:table:ModifiedAndSavedVarnames')

% Rename input
Folder1 = Data.Folder1;
Folder2 = Data.Folder2;
FileNameEC = Data.FileNameEC;
Area = Data.Area;
AreaErr = Data.AreaErr;
ESet = Data.ESet;
VarName = Data.VarName;
VCat = Data.VCat;

% Main file folder
Location = {'...\Data Chapter 4 Ethylene'};

% Some constants
Vm = 22.4;      % Molar volume (ml/mmol)
F = 96485;      % Faraday constant (C/mol)
MwAc = 59.04;   % Molecular weight acetate (g/mol)
MwProp = 60.10; % Molecular weight 1-propanol (g/mol)

% Intercept and slope of calibration curve for H2 (Nov22)
b0H2 = -2025.09592;
b1H2 = 294.54225;

% Intercept and slope of calibration cure for FA (Aug22)
b0FA = 125.67316;
b1FA = 464.91233;

% Intercept and slope of calibration curve for acetate (Nov22)
b0Ac = 1319.73;
b1Ac = 572.74;

% Intercept and slope of calibration curve for 1-propanol (Nov22)
b0Prop = 421.06;
b1Prop = 814.03;

%% Calibration input

% H2
CalH2Data = importdata('...\Data Chapter 4 Ethylene\Matlab\CalH2DataNov22.txt');
XCalH2 = [1000*ones(12,1); 1500*ones(12,1); 2500*ones(12,1); 5000*ones(11,1); 8000*ones(12,1); 10000*ones(12,1); 15000*ones(12,1); 20000*ones(12,1)]';
YCalH2 = [CalH2Data.data(:,1); CalH2Data.data(:,2); CalH2Data.data(:,3); CalH2Data.data(:,4); CalH2Data.data(:,5); CalH2Data.data(:,6); CalH2Data.data(:,7);CalH2Data.data(:,8)]';

% FA calibration Aug22
CalFAAugData = importdata('...\Data Chapter 4 Ethylene\Matlab\CalFADataAug22.txt');
XCalFAAug22 = [1*ones(2,1);10*ones(2,1); 50*ones(2,1); 100*ones(2,1); 150*ones(2,1)]';
YCalFAAug22 = [CalFAAugData.data(:,1); CalFAAugData.data(:,2); CalFAAugData.data(:,3); CalFAAugData.data(:,4); CalFAAugData.data(:,5)]';

% Acetate
CalAcData = importdata('...\Data Chapter 4 Ethylene\Matlab\CalAcDataNov22.txt');
XCalAcNov22 = [2.003*ones(2,1); 5.006*ones(2,1); 10.0126*ones(2,1); 15.019*ones(2,1); 20.025*ones(2,1); 40*ones(2,1); 60*ones(2,1); 80*ones(2,1); 100*ones(2,1)]';
YCalAcNov22 = [CalAcData.data(:,1);CalAcData.data(:,2); CalAcData.data(:,3); CalAcData.data(:,4); CalAcData.data(:,5); CalAcData.data(:,6); CalAcData.data(:,7); CalAcData.data(:,8); CalAcData.data(:,9)]';

% 1-Propanol
XCalPropNov22 = [10 10 25 25 50 50 75 75 100 100];
YCalPropNov22 = [8609.7 7744.3 20914.3 20714.2 40365.6 40061.3 65905.9 63187.3 80206.1 79795.1];

%% TCD file path
FileNameTCD = {'\Anne20220208-TCD.txt'};
FilePathTCD = append(Location, Folder1, Folder2, FileNameTCD);
FilePathTCD = cell2mat(FilePathTCD);  % Convert cell array with file paths to matrix

%% FID data file path
FileNameFID = {'\Anne20220208-FID.txt'};
FilePathFID = append(Location, Folder1, Folder2, FileNameFID);
FilePathFID = cell2mat(FilePathFID);  % Convert cell array with file paths to matrix

%% FA data file path
FilePathFA = append(Location, Folder1, '\FA areas.txt');
FilePathFA = cell2mat(FilePathFA);  % Convert cell array with file paths to matrix
LoadDataFA = importdata(FilePathFA);

%% Acetate data file path
FilePathAc = append(Location, Folder1, '\acetate areas.txt');
FilePathAc = cell2mat(FilePathAc);  % Convert cell array with file paths to matrix
LoadDataAc = importdata(FilePathAc);

%% 1-Propanol data file path
FilePathProp = append(Location, Folder1, '\propanol areas.txt');
FilePathProp = cell2mat(FilePathProp);  % Convert cell array with file paths to matrix
LoadDataProp = importdata(FilePathProp);

%% File path resistance values
FilePathR =  append(Location, Folder1, '\resistances.txt');
FilePathR = cell2mat(FilePathR);  % Convert cell array with file paths to matrix
LoadDataResistance = importdata(FilePathR);

%% Calculations

for j = 1:length(ESet)
    FlowRate = Data.FlowRate(j);

    FilePathEC = append(Location, Folder1, Folder2(j,:), FileNameEC(j,:)); % Retrieve file path potentiostat data
    FilePathEC = cell2mat(FilePathEC);  % Convert cell array with file paths to matrix

    %% Load data from potentiostat
    LoadDataEC = importdata(FilePathEC);

    % LoadDataEC is a struct containing fields for every investigated potential
    % The columns refer to: time/s, <I>/mA, Ewe/V, <Ece>/V, dQ/C, Rcmp/Ohm,
    % P/W, Ewe-Ece/V
    Data.(VarName(j,:)).EC = LoadDataEC.data;

    % Recalculate time (time does not start at t=0)
    tElapsed = Data.(VarName(j,:)).EC(:,1);
    tStart = Data.(VarName(j,:)).EC(1,1);
    Data.(VarName(j,:)).t = tElapsed - tStart;

    % Total current (mA)
    Data.(VarName(j,:)).I = Data.(VarName(j,:)).EC(:,2);

    % Calculate current density (mA cm^-2)
    Data.(VarName(j,:)).CD = Data.(VarName(j,:)).EC(:,2)/Area(j);

    % Calculate average current (density) over the last 30 minutes of the
    % experiment
    findTime = find(Data.(VarName(j,:)).t >= 30*60);

    q1 = size(findTime);        % Length of time vector from 30 mins to end of experiment

    % Average current and estimation of the standard deviation
    Data.Iavg30mins(j,1) = mean(Data.(VarName(j,:)).I(findTime(:,1),1));
    Data.Iavg30mins(j,2) = sqrt(sum((Data.(VarName(j,:)).I(findTime(:,1),1)-Data.Iavg30mins(j,1)).^2)/(q1(1,1)-1));

    % Average current denisty and estimation of the standard deviation
    Data.CDavg30mins(j,1) = mean(Data.(VarName(j,:)).CD(findTime(:,1),1));
    Data.CDavg30mins(j,2) = sqrt((Data.Iavg30mins(j,2)/Data.Iavg30mins(j,1))^2 + ((AreaErr(j)/Area(j))^2))*Data.CDavg30mins(j,1);

    %% Obtain H2 current and H2 partial current density
    LoadDataTCD = readtable(FilePathTCD(j,:));
    LoadDataTCD = LoadDataTCD(:,7:end);
    Data.(VarName(j,:)).TCD = LoadDataTCD;

    TCDArea = table2array(Data.(VarName(j,:)).TCD);

    % Calculate average area of injections 5, 6 and 7
    Data.TCD_avg(j,:) = mean(TCDArea(5:end,1:5));

    % Obtain H2 concentration and error in H2 concentration
    Width = size(Data.(VarName(j,:)).TCD(:,1));
    q = Width(1,1)-4; % Determine amount of values used to calculate the average
    Data.ConcAndErr(j,:) = CalcErr(XCalH2, YCalH2, q,b0H2,b1H2,Data.TCD_avg(j,1));

    % Calculate corresponding current
    Data.IH2(j,:) = ((((Data.ConcAndErr(j,:))/10000/100)*(FlowRate/Vm))*-2*F);

    % Calculate corresponding current density
    Data.CDH2(j,1) = Data.IH2(j,1)/Area(j);

    % Calculate error in current density
    Data.CDH2(j,2) = sqrt(((Data.IH2(j,2))/Data.IH2(j,1))^2+(AreaErr(j)/Area(j))^2)*(Data.CDH2(j,1));

    %% Obtain hydrocarbons current and hydrocarbons partial current density
    LoadDataFID = readtable(FilePathFID(j,:));
    LoadDataFID = LoadDataFID(:,7:end);
    Data.(VarName(j,:)).FID = LoadDataFID;

    % Hydrocarbon calibration
    XCal = [933 931 957 958 967];       % ppm methane, ethylene, ethane, propylene, propane
    YCal = [20915; 39461; 41438; 60924; 61100]; % Peak area calibration
    StDevCal = [676.6660416	1266.426098	1330.122377	2000.205671	1940.911142]';  % Standard deviation of peak area
    n = [6 8 10 12 14];      % Amount of electrons involved for converting CO in methane, ethene, ethane, etc.

    FIDArea = table2array(Data.(VarName(j,:)).FID);

    % Calculate average area of injections 5, 6 and 7
    Data.FID_avg(j,:) = mean(FIDArea(5:end,1:5));

    % Determine amount of values used to calculate the average of
    % injections
    Width = size(Data.(VarName(j,:)).FID(:,1));
    q = Width(1,1)-4; 
    
    % Estimate standard deviation of injections 5, 6, 7
    Data.FID_avgstdev(j,:) = sqrt(sum((FIDArea(5:end,1:5)-Data.FID_avg(j,:)).^2)/(q-1));

    % Calculate current (mA) and current density (mA cm^-2)
    Data.FIDppm(j,:) = Data.FID_avg(j,:).*XCal./YCal';
    Data.FID_I(j,:) = ((((Data.FIDppm(j,:))/10000/100)*(FlowRate/Vm)).*-n*F);
    Data.FID_CD(j,:) = Data.FID_I(j,:)/Area(j);

    % Estimate error in current and current density
    for i = 1:5
        Data.FIDppmErr(j,i) = Data.FIDppm(j,i)*sqrt((Data.FID_avgstdev(j,i)/Data.FID_avg(j,i)).^2+(StDevCal(i,:)/YCal(i,:)).^2);
        Data.FID_I_Err(j,i)=((((Data.FIDppmErr(j,i))/10000/100)*(FlowRate/Vm)).*-n(i)*F);

        Data.FID_CD_Err(j,i) = sqrt(((Data.FID_I_Err(j,i))/Data.FID_I(j,i))^2+(AreaErr(j)/Area(j))^2)*(Data.FID_CD(j,i));
    end

    %% Obtain formate concentration in catholyte
    Data.FA(j,:) = LoadDataFA.data(j,:);

    Data.ConcAndErrFACat(j,:) = CalcErr(XCalFAAug22,YCalFAAug22,1,b0FA,b1FA,Data.FA(j,1));   % FA concentration in catholyte + error

    Data.ConcAndErrFA(j,1) = Data.ConcAndErrFACat(j,1);
    Data.ConcAndErrFA(j,2) = Data.ConcAndErrFACat(j,2);

    % The formation of formate most likely occurs through a chemical rather
    % than an electrochemical reaction. For this reason only the
    % concentration FA is calculated

    %% Obtain acetate partial current density

    Data.Ac(j,:) = LoadDataAc.data(j,:);

    Data.ConcAndErrAcCat(j,:) = CalcErr(XCalAcNov22,YCalAcNov22,1,b0Ac,b1Ac,Data.Ac(j,1));   % Ac concentration in catholyte + error
    Data.ConcAndErrAcAn(j,:) = CalcErr(XCalAcNov22,YCalAcNov22,1,b0Ac,b1Ac,Data.Ac(j,2));    % Ac concentration in anolyte + error

    if Data.ConcAndErrAcCat(j,1) > 0  & Data.ConcAndErrAcAn(j,1) > 0
        Data.ConcAndErrAc(j,1) = Data.ConcAndErrAcCat(j,1) + Data.ConcAndErrAcAn(j,1);
        Data.ConcAndErrAc(j,2) = sqrt((Data.ConcAndErrAcCat(j,2))^2 + (Data.ConcAndErrAcAn(j,2))^2);
    elseif Data.ConcAndErrAcCat(j,1) > 0
        Data.ConcAndErrAc(j,1) = Data.ConcAndErrAcCat(j,1);
        Data.ConcAndErrAc(j,2) = Data.ConcAndErrAcCat(j,2);
    else    % This means only Data.ConcAndErrAcAn(j,:) > 0
        Data.ConcAndErrAc(j,1) = Data.ConcAndErrAcAn(j,1);
        Data.ConcAndErrAc(j,2) = Data.ConcAndErrAcAn(j,2);
    end

    % Calculate corresponding current
    Data.IAc(j,:) = (((Data.ConcAndErrAc(j,:))/1000/MwAc)*VCat(j)*-4*F)/3600*1000;

    % Calculate corresponding current density
    Data.CDAc(j,1) = Data.IAc(j,1)/Area(j);

    % Calculate error in current density
    Data.CDAc(j,2) = sqrt(((Data.IAc(j,2))/Data.IAc(j,1))^2+(AreaErr(j)/Area(j))^2)*(Data.CDAc(j,1));

    %% Obtain 1-propanol partial current density

    Data.Prop(j,:) = LoadDataProp.data(j,:);

    Data.ConcAndErrPropCat(j,:) = CalcErr(XCalPropNov22,YCalPropNov22,1,b0Prop,b1Prop,Data.Prop(j,1));   % Prop concentration in catholyte + error
    Data.ConcAndErrPropAn(j,:) = CalcErr(XCalPropNov22,YCalPropNov22,1,b0Prop,b1Prop,Data.Prop(j,2));    % Prop concentration in anolyte + error

    if Data.ConcAndErrPropCat(j,1) > 0  & Data.ConcAndErrPropAn(j,1) > 0
        Data.ConcAndErrProp(j,1) = Data.ConcAndErrPropCat(j,1) + Data.ConcAndErrPropAn(j,1);
        Data.ConcAndErrProp(j,2) = sqrt((Data.ConcAndErrPropCat(j,2))^2 + (Data.ConcAndErrPropAn(j,2))^2);
    elseif Data.ConcAndErrPropCat(j,1) > 0
        Data.ConcAndErrProp(j,1) = Data.ConcAndErrPropCat(j,1);
        Data.ConcAndErrProp(j,2) = Data.ConcAndErrPropCat(j,2);
    else    % This means only Data.ConcAndErrPropAn(j,:) > 0
        Data.ConcAndErrProp(j,1) = Data.ConcAndErrPropAn(j,1);
        Data.ConcAndErrProp(j,2) = Data.ConcAndErrPropAn(j,2);
    end

    % Calculate corresponding current
    Data.IProp(j,:) = (((Data.ConcAndErrProp(j,:))/1000/MwProp)*VCat(j)*-12*F)/3600*1000;

    % Calculate corresponding current density
    Data.CDProp(j,1) = Data.IProp(j,1)/Area(j);

    Data.CDProp(j,2) = sqrt(((Data.IProp(j,2))/Data.IProp(j,1))^2+(AreaErr(j)/Area(j))^2)*(Data.CDProp(j,1));

    %% Calculate actual potential at working electrode and error in the potential

    Data.R(j,:) = LoadDataResistance(j,:);

    RAvg = mean(Data.R(j,:));

    % Actual potentail
    Data.Ereal(j,1) = ESet(j)- mean((Data.(VarName(j,:)).EC(findTime(:,1),2))/1000)*RAvg;

    % Error calculations
    RErr = Data.R(j,1) - RAvg;

    %Error in potential
    Data.Ereal(j,2) =  mean((Data.(VarName(j,:)).EC(findTime(:,1),2))/1000)*RErr;

    %% Calculate current that is not assigned

    % Convert all NaN in 0
    IndexAC = isnan(Data.CDAc);
    Data.CDAc(IndexAC) = 0;

    IndexProp = isnan(Data.CDProp);
    Data.CDProp(IndexProp) = 0;

    b = size(Data.FID_CD);

    for i = 1:b(1,2)
        IndexFID = isnan(Data.FID_CD(:,i));
        Data.FID_CD(IndexFID,i) = 0;
    end

    Data.CDUnaccounted(j,1) = Data.CDavg30mins(j,1) - Data.CDH2(j,1) - Data.CDAc(j,1) - Data.CDProp(j,1) - Data.FID_CD(j,2) - Data.FID_CD(j,3) - Data.FID_CD(j,4);
end
end